﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/Array.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/lambda/LambdaRequest.h>
#include <aws/lambda/Lambda_EXPORTS.h>
#include <aws/lambda/model/Architecture.h>
#include <aws/lambda/model/FunctionVersionLatestPublished.h>

#include <utility>

namespace Aws {
namespace Lambda {
namespace Model {

/**
 */
class UpdateFunctionCodeRequest : public LambdaRequest {
 public:
  AWS_LAMBDA_API UpdateFunctionCodeRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "UpdateFunctionCode"; }

  AWS_LAMBDA_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>The name or ARN of the Lambda function.</p> <p class="title"> <b>Name
   * formats</b> </p> <ul> <li> <p> <b>Function name</b> –
   * <code>my-function</code>.</p> </li> <li> <p> <b>Function ARN</b> –
   * <code>arn:aws:lambda:us-west-2:123456789012:function:my-function</code>.</p>
   * </li> <li> <p> <b>Partial ARN</b> –
   * <code>123456789012:function:my-function</code>.</p> </li> </ul> <p>The length
   * constraint applies only to the full ARN. If you specify only the function name,
   * it is limited to 64 characters in length.</p>
   */
  inline const Aws::String& GetFunctionName() const { return m_functionName; }
  inline bool FunctionNameHasBeenSet() const { return m_functionNameHasBeenSet; }
  template <typename FunctionNameT = Aws::String>
  void SetFunctionName(FunctionNameT&& value) {
    m_functionNameHasBeenSet = true;
    m_functionName = std::forward<FunctionNameT>(value);
  }
  template <typename FunctionNameT = Aws::String>
  UpdateFunctionCodeRequest& WithFunctionName(FunctionNameT&& value) {
    SetFunctionName(std::forward<FunctionNameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The base64-encoded contents of the deployment package. Amazon Web Services
   * SDK and CLI clients handle the encoding for you. Use only with a function
   * defined with a .zip file archive deployment package.</p>
   */
  inline const Aws::Utils::CryptoBuffer& GetZipFile() const { return m_zipFile; }
  inline bool ZipFileHasBeenSet() const { return m_zipFileHasBeenSet; }
  template <typename ZipFileT = Aws::Utils::CryptoBuffer>
  void SetZipFile(ZipFileT&& value) {
    m_zipFileHasBeenSet = true;
    m_zipFile = std::forward<ZipFileT>(value);
  }
  template <typename ZipFileT = Aws::Utils::CryptoBuffer>
  UpdateFunctionCodeRequest& WithZipFile(ZipFileT&& value) {
    SetZipFile(std::forward<ZipFileT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>An Amazon S3 bucket in the same Amazon Web Services Region as your function.
   * The bucket can be in a different Amazon Web Services account. Use only with a
   * function defined with a .zip file archive deployment package.</p>
   */
  inline const Aws::String& GetS3Bucket() const { return m_s3Bucket; }
  inline bool S3BucketHasBeenSet() const { return m_s3BucketHasBeenSet; }
  template <typename S3BucketT = Aws::String>
  void SetS3Bucket(S3BucketT&& value) {
    m_s3BucketHasBeenSet = true;
    m_s3Bucket = std::forward<S3BucketT>(value);
  }
  template <typename S3BucketT = Aws::String>
  UpdateFunctionCodeRequest& WithS3Bucket(S3BucketT&& value) {
    SetS3Bucket(std::forward<S3BucketT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The Amazon S3 key of the deployment package. Use only with a function defined
   * with a .zip file archive deployment package.</p>
   */
  inline const Aws::String& GetS3Key() const { return m_s3Key; }
  inline bool S3KeyHasBeenSet() const { return m_s3KeyHasBeenSet; }
  template <typename S3KeyT = Aws::String>
  void SetS3Key(S3KeyT&& value) {
    m_s3KeyHasBeenSet = true;
    m_s3Key = std::forward<S3KeyT>(value);
  }
  template <typename S3KeyT = Aws::String>
  UpdateFunctionCodeRequest& WithS3Key(S3KeyT&& value) {
    SetS3Key(std::forward<S3KeyT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>For versioned objects, the version of the deployment package object to
   * use.</p>
   */
  inline const Aws::String& GetS3ObjectVersion() const { return m_s3ObjectVersion; }
  inline bool S3ObjectVersionHasBeenSet() const { return m_s3ObjectVersionHasBeenSet; }
  template <typename S3ObjectVersionT = Aws::String>
  void SetS3ObjectVersion(S3ObjectVersionT&& value) {
    m_s3ObjectVersionHasBeenSet = true;
    m_s3ObjectVersion = std::forward<S3ObjectVersionT>(value);
  }
  template <typename S3ObjectVersionT = Aws::String>
  UpdateFunctionCodeRequest& WithS3ObjectVersion(S3ObjectVersionT&& value) {
    SetS3ObjectVersion(std::forward<S3ObjectVersionT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>URI of a container image in the Amazon ECR registry. Do not use for a
   * function defined with a .zip file archive.</p>
   */
  inline const Aws::String& GetImageUri() const { return m_imageUri; }
  inline bool ImageUriHasBeenSet() const { return m_imageUriHasBeenSet; }
  template <typename ImageUriT = Aws::String>
  void SetImageUri(ImageUriT&& value) {
    m_imageUriHasBeenSet = true;
    m_imageUri = std::forward<ImageUriT>(value);
  }
  template <typename ImageUriT = Aws::String>
  UpdateFunctionCodeRequest& WithImageUri(ImageUriT&& value) {
    SetImageUri(std::forward<ImageUriT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to true to publish a new version of the function after updating the code.
   * This has the same effect as calling <a>PublishVersion</a> separately.</p>
   */
  inline bool GetPublish() const { return m_publish; }
  inline bool PublishHasBeenSet() const { return m_publishHasBeenSet; }
  inline void SetPublish(bool value) {
    m_publishHasBeenSet = true;
    m_publish = value;
  }
  inline UpdateFunctionCodeRequest& WithPublish(bool value) {
    SetPublish(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Set to true to validate the request parameters and access permissions without
   * modifying the function code.</p>
   */
  inline bool GetDryRun() const { return m_dryRun; }
  inline bool DryRunHasBeenSet() const { return m_dryRunHasBeenSet; }
  inline void SetDryRun(bool value) {
    m_dryRunHasBeenSet = true;
    m_dryRun = value;
  }
  inline UpdateFunctionCodeRequest& WithDryRun(bool value) {
    SetDryRun(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Update the function only if the revision ID matches the ID that's specified.
   * Use this option to avoid modifying a function that has changed since you last
   * read it.</p>
   */
  inline const Aws::String& GetRevisionId() const { return m_revisionId; }
  inline bool RevisionIdHasBeenSet() const { return m_revisionIdHasBeenSet; }
  template <typename RevisionIdT = Aws::String>
  void SetRevisionId(RevisionIdT&& value) {
    m_revisionIdHasBeenSet = true;
    m_revisionId = std::forward<RevisionIdT>(value);
  }
  template <typename RevisionIdT = Aws::String>
  UpdateFunctionCodeRequest& WithRevisionId(RevisionIdT&& value) {
    SetRevisionId(std::forward<RevisionIdT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The instruction set architecture that the function supports. Enter a string
   * array with one of the valid values (arm64 or x86_64). The default value is
   * <code>x86_64</code>.</p>
   */
  inline const Aws::Vector<Architecture>& GetArchitectures() const { return m_architectures; }
  inline bool ArchitecturesHasBeenSet() const { return m_architecturesHasBeenSet; }
  template <typename ArchitecturesT = Aws::Vector<Architecture>>
  void SetArchitectures(ArchitecturesT&& value) {
    m_architecturesHasBeenSet = true;
    m_architectures = std::forward<ArchitecturesT>(value);
  }
  template <typename ArchitecturesT = Aws::Vector<Architecture>>
  UpdateFunctionCodeRequest& WithArchitectures(ArchitecturesT&& value) {
    SetArchitectures(std::forward<ArchitecturesT>(value));
    return *this;
  }
  inline UpdateFunctionCodeRequest& AddArchitectures(Architecture value) {
    m_architecturesHasBeenSet = true;
    m_architectures.push_back(value);
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>The ARN of the Key Management Service (KMS) customer managed key that's used
   * to encrypt your function's .zip deployment package. If you don't provide a
   * customer managed key, Lambda uses an Amazon Web Services managed key.</p>
   */
  inline const Aws::String& GetSourceKMSKeyArn() const { return m_sourceKMSKeyArn; }
  inline bool SourceKMSKeyArnHasBeenSet() const { return m_sourceKMSKeyArnHasBeenSet; }
  template <typename SourceKMSKeyArnT = Aws::String>
  void SetSourceKMSKeyArn(SourceKMSKeyArnT&& value) {
    m_sourceKMSKeyArnHasBeenSet = true;
    m_sourceKMSKeyArn = std::forward<SourceKMSKeyArnT>(value);
  }
  template <typename SourceKMSKeyArnT = Aws::String>
  UpdateFunctionCodeRequest& WithSourceKMSKeyArn(SourceKMSKeyArnT&& value) {
    SetSourceKMSKeyArn(std::forward<SourceKMSKeyArnT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Specifies where to publish the function version or configuration.</p>
   */
  inline FunctionVersionLatestPublished GetPublishTo() const { return m_publishTo; }
  inline bool PublishToHasBeenSet() const { return m_publishToHasBeenSet; }
  inline void SetPublishTo(FunctionVersionLatestPublished value) {
    m_publishToHasBeenSet = true;
    m_publishTo = value;
  }
  inline UpdateFunctionCodeRequest& WithPublishTo(FunctionVersionLatestPublished value) {
    SetPublishTo(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_functionName;
  bool m_functionNameHasBeenSet = false;

  Aws::Utils::CryptoBuffer m_zipFile{};
  bool m_zipFileHasBeenSet = false;

  Aws::String m_s3Bucket;
  bool m_s3BucketHasBeenSet = false;

  Aws::String m_s3Key;
  bool m_s3KeyHasBeenSet = false;

  Aws::String m_s3ObjectVersion;
  bool m_s3ObjectVersionHasBeenSet = false;

  Aws::String m_imageUri;
  bool m_imageUriHasBeenSet = false;

  bool m_publish{false};
  bool m_publishHasBeenSet = false;

  bool m_dryRun{false};
  bool m_dryRunHasBeenSet = false;

  Aws::String m_revisionId;
  bool m_revisionIdHasBeenSet = false;

  Aws::Vector<Architecture> m_architectures;
  bool m_architecturesHasBeenSet = false;

  Aws::String m_sourceKMSKeyArn;
  bool m_sourceKMSKeyArnHasBeenSet = false;

  FunctionVersionLatestPublished m_publishTo{FunctionVersionLatestPublished::NOT_SET};
  bool m_publishToHasBeenSet = false;
};

}  // namespace Model
}  // namespace Lambda
}  // namespace Aws
