﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSMap.h>
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/mgn/MgnRequest.h>
#include <aws/mgn/Mgn_EXPORTS.h>
#include <aws/mgn/model/ConnectorSsmCommandConfig.h>

#include <utility>

namespace Aws {
namespace mgn {
namespace Model {

/**
 */
class CreateConnectorRequest : public MgnRequest {
 public:
  AWS_MGN_API CreateConnectorRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "CreateConnector"; }

  AWS_MGN_API Aws::String SerializePayload() const override;

  ///@{
  /**
   * <p>Create Connector request name.</p>
   */
  inline const Aws::String& GetName() const { return m_name; }
  inline bool NameHasBeenSet() const { return m_nameHasBeenSet; }
  template <typename NameT = Aws::String>
  void SetName(NameT&& value) {
    m_nameHasBeenSet = true;
    m_name = std::forward<NameT>(value);
  }
  template <typename NameT = Aws::String>
  CreateConnectorRequest& WithName(NameT&& value) {
    SetName(std::forward<NameT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Create Connector request SSM command config.</p>
   */
  inline const ConnectorSsmCommandConfig& GetSsmCommandConfig() const { return m_ssmCommandConfig; }
  inline bool SsmCommandConfigHasBeenSet() const { return m_ssmCommandConfigHasBeenSet; }
  template <typename SsmCommandConfigT = ConnectorSsmCommandConfig>
  void SetSsmCommandConfig(SsmCommandConfigT&& value) {
    m_ssmCommandConfigHasBeenSet = true;
    m_ssmCommandConfig = std::forward<SsmCommandConfigT>(value);
  }
  template <typename SsmCommandConfigT = ConnectorSsmCommandConfig>
  CreateConnectorRequest& WithSsmCommandConfig(SsmCommandConfigT&& value) {
    SetSsmCommandConfig(std::forward<SsmCommandConfigT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Create Connector request SSM instance ID.</p>
   */
  inline const Aws::String& GetSsmInstanceID() const { return m_ssmInstanceID; }
  inline bool SsmInstanceIDHasBeenSet() const { return m_ssmInstanceIDHasBeenSet; }
  template <typename SsmInstanceIDT = Aws::String>
  void SetSsmInstanceID(SsmInstanceIDT&& value) {
    m_ssmInstanceIDHasBeenSet = true;
    m_ssmInstanceID = std::forward<SsmInstanceIDT>(value);
  }
  template <typename SsmInstanceIDT = Aws::String>
  CreateConnectorRequest& WithSsmInstanceID(SsmInstanceIDT&& value) {
    SetSsmInstanceID(std::forward<SsmInstanceIDT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Create Connector request tags.</p>
   */
  inline const Aws::Map<Aws::String, Aws::String>& GetTags() const { return m_tags; }
  inline bool TagsHasBeenSet() const { return m_tagsHasBeenSet; }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  void SetTags(TagsT&& value) {
    m_tagsHasBeenSet = true;
    m_tags = std::forward<TagsT>(value);
  }
  template <typename TagsT = Aws::Map<Aws::String, Aws::String>>
  CreateConnectorRequest& WithTags(TagsT&& value) {
    SetTags(std::forward<TagsT>(value));
    return *this;
  }
  template <typename TagsKeyT = Aws::String, typename TagsValueT = Aws::String>
  CreateConnectorRequest& AddTags(TagsKeyT&& key, TagsValueT&& value) {
    m_tagsHasBeenSet = true;
    m_tags.emplace(std::forward<TagsKeyT>(key), std::forward<TagsValueT>(value));
    return *this;
  }
  ///@}
 private:
  Aws::String m_name;
  bool m_nameHasBeenSet = false;

  ConnectorSsmCommandConfig m_ssmCommandConfig;
  bool m_ssmCommandConfigHasBeenSet = false;

  Aws::String m_ssmInstanceID;
  bool m_ssmInstanceIDHasBeenSet = false;

  Aws::Map<Aws::String, Aws::String> m_tags;
  bool m_tagsHasBeenSet = false;
};

}  // namespace Model
}  // namespace mgn
}  // namespace Aws
